@extends('layouts.millage')

@section('title', 'Trade')

@section('content')
<div class="h-full md:p-4 py-4 bg-theme-tertiary">
    <div class="w-full mx-auto px-4 sm:px-6 lg:px-2">
        {{-- Welcome Section --}}
        <div class="bg-theme-secondary rounded-xl md:p-8 p-6 mb-6">
            <div class="w-full">
                <h1 class="text-2xl font-medium mb-3">Welcome to Trading</h1>
                <p class="text-gray-400 mb-6">Experience real-time trading with advanced tools and analytics. Start trading with confidence using our intuitive platform.</p>
                
                <div class="flex flex-wrap gap-4">
                    <a href="{{ route('user.trading.trade') }}" 
                        class="inline-flex items-center gap-2 bg-gradient-theme-primary text-black font-medium px-6 py-3 rounded-xl hover:brightness-110 transition-all"
                    >
                        <i class="fas fa-chart-line"></i>
                        Start Trading
                    </a>
                    
                    <a href="{{ route('user.trading.history') }}" 
                        class="inline-flex items-center gap-2 bg-theme-tertiary text-white font-medium px-6 py-3 rounded-xl hover:bg-theme-tertiary/80 transition-all"
                    >
                        <i class="fas fa-history"></i>
                        Trading History
                    </a>
                </div>
            </div>
        </div>

        {{-- Trading Stats Grid --}}
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
            {{-- Total Trades --}}
            <div class="bg-theme-secondary rounded-xl p-6">
                <div class="flex items-start justify-between">
                    <div>
                        <p class="text-sm text-gray-400">Total Trades</p>
                        <p class="text-2xl font-medium mt-1">{{ number_format($stats['total_trades']) }}</p>
                    </div>
                    <div class="w-10 h-10 flex items-center justify-center rounded-xl bg-blue-500/10">
                        <i class="fas fa-chart-bar text-blue-500"></i>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-800/50">
                    <span class="text-sm text-gray-400">Lifetime trades</span>
                </div>
            </div>

            {{-- Win Rate --}}
            <div class="bg-theme-secondary rounded-xl p-6">
                <div class="flex items-start justify-between">
                    <div>
                        <p class="text-sm text-gray-400">Win Rate</p>
                        <p class="text-2xl font-medium mt-1">{{ number_format($stats['win_rate'], 1) }}%</p>
                    </div>
                    <div class="w-10 h-10 flex items-center justify-center rounded-xl bg-green-500/10">
                        <i class="fas fa-chart-line text-green-500"></i>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-800/50">
                    <span class="text-sm {{ $stats['win_rate'] > 50 ? 'text-green-500' : 'text-gray-400' }}">
                        {{ $stats['win_rate'] > 50 ? 'Above average' : 'Keep improving' }}
                    </span>
                </div>
            </div>

            {{-- Total Profit --}}
            <div class="bg-theme-secondary rounded-xl p-6">
                <div class="flex items-start justify-between">
                    <div>
                        <p class="text-sm text-gray-400">Total Profit</p>
                        <p class="text-2xl font-medium mt-1 {{ $stats['total_profit'] >= 0 ? 'text-green-500' : 'text-red-500' }}">
                            {{ $stats['total_profit'] >= 0 ? '+' : '' }}${{ number_format($stats['total_profit'], 2) }}
                        </p>
                    </div>
                    <div class="w-10 h-10 flex items-center justify-center rounded-xl bg-theme-primary/10">
                        <i class="fas fa-dollar-sign text-theme-primary"></i>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-800/50">
                    <span class="text-sm text-gray-400">Overall performance</span>
                </div>
            </div>

            {{-- Account Balance --}}
            <div class="bg-theme-secondary rounded-xl p-6">
                <div class="flex items-start justify-between">
                    <div>
                        <p class="text-sm text-gray-400">Available Balance</p>
                        <p class="text-2xl font-medium mt-1">${{ number_format($user->account_bal, 2) }}</p>
                    </div>
                    <div class="w-10 h-10 flex items-center justify-center rounded-xl bg-yellow-500/10">
                        <i class="fas fa-wallet text-yellow-500"></i>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-800/50">
                    <span class="text-sm text-gray-400">Trading balance</span>
                </div>
            </div>
        </div>

       {{-- Trading Pairs Section --}}
<div class="relative">
    <h2 class="text-xl font-medium mb-4">Available Trading Pairs</h2>
    
    <div id="tradingPairsCarousel" class="relative overflow-hidden">
        <div class="trading-pairs-wrapper flex transition-transform duration-300 ease-in-out" style="transform: translateX(0)">
            @foreach($pairs as $pair)
                <div class="trading-pair-slide flex-shrink-0 w-full md:w-1/2 lg:w-1/3 p-2">
                    <div class="bg-theme-secondary rounded-xl p-4 hover:border-2 hover:border-theme-primary/30 transition-all cursor-pointer group"
                         onclick="window.location.href='{{ route('user.trading.trade', ['pair' => str_replace('/', '', $pair['symbol'])]) }}'">
                        <div class="flex items-center gap-4">
                            <div class="w-12 h-12 flex items-center justify-center bg-gray-800/50 rounded-xl">
                                <img src="{{ $pair['icon'] }}" alt="{{ $pair['name'] }} Logo" class="pair-icon w-6 h-6">
                            </div>
                            <div>
                                <h3 class="font-medium">{{ $pair['name'] }}</h3>
                                <p class="text-sm text-gray-400">{{ $pair['symbol'] }}</p>
                            </div>
                            <div class="ml-auto">
                                <div class="w-8 h-8 flex items-center justify-center rounded-full bg-theme-primary/10 group-hover:bg-theme-primary/20 transition-colors">
                                    <i class="fas fa-arrow-right text-theme-primary"></i>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mt-4 pt-4 border-t border-gray-800/50">
                            <div class="flex items-center justify-between">
                                <span class="text-sm text-gray-400">Current Price</span>
                                <span class="text-sm font-mono price-{{ str_replace('/', '', $pair['symbol']) }}" data-price="{{ $pair['price'] ?? 0 }}">
                                    ${{ number_format($pair['price'] ?? 0, 2) }}
                                </span>
                            </div>
                            <div class="flex items-center justify-between mt-2">
                                <span class="text-sm text-gray-400">24h Change</span>
                                <span class="text-sm change-{{ str_replace('/', '', $pair['symbol']) }}" 
                                      data-change="{{ $pair['change_24h'] ?? 0 }}"
                                      class="text-sm {{ ($pair['change_24h'] ?? 0) >= 0 ? 'text-green-500' : 'text-red-500' }} change-{{ str_replace('/', '', $pair['symbol']) }}">
                                    {{ ($pair['change_24h'] ?? 0) >= 0 ? '+' : '' }}{{ number_format($pair['change_24h'] ?? 0, 2) }}%
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            @endforeach
        </div>
    </div>

    {{-- Navigation Buttons --}}
    <div class="absolute top-1/2 -translate-y-1/2 w-full flex justify-between pointer-events-none">
        <button onclick="scrollTradingPairs(-1)" class="trading-pair-nav-btn bg-theme-primary/20 text-theme-primary p-2 rounded-full opacity-0 group-hover:opacity-100 transition-all pointer-events-auto">
            <i class="fas fa-chevron-left"></i>
        </button>
        <button onclick="scrollTradingPairs(1)" class="trading-pair-nav-btn bg-theme-primary/20 text-theme-primary p-2 rounded-full opacity-0 group-hover:opacity-100 transition-all pointer-events-auto">
            <i class="fas fa-chevron-right"></i>
        </button>
    </div>

    {{-- Pagination Dots 
    <div class="flex justify-center mt-4 space-x-2">
        @foreach($pairs as $index => $pair)
            <button class="trading-pair-dot w-2 h-2 rounded-full bg-gray-600 hover:bg-theme-primary transition-colors" 
                    onclick="goToPair({{ $index }})"></button>
        @endforeach
    </div> --}}
</div>

        {{-- Recent Trades --}}
        <div class="bg-theme-secondary rounded-xl overflow-hidden mt-4">
            <div class="p-4 border-b border-gray-800/50 flex items-center justify-between">
                <h2 class="text-lg font-medium">Recent Trades</h2>
                <a href="{{ route('user.trading.history') }}" class="text-theme-primary text-sm hover:underline">
                    View All
                </a>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead class="bg-theme-tertiary">
                        <tr>
                            <th class="px-4 py-3 text-left text-sm font-medium text-gray-400">Time</th>
                            <th class="px-4 py-3 text-left text-sm font-medium text-gray-400">Pair</th>
                            <th class="px-4 py-3 text-left text-sm font-medium text-gray-400">Type</th>
                            <th class="px-4 py-3 text-left text-sm font-medium text-gray-400">Amount</th>
                            <th class="px-4 py-3 text-left text-sm font-medium text-gray-400">Result</th>
                            <th class="px-4 py-3 text-left text-sm font-medium text-gray-400">Profit/Loss</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-800/50">
                        @forelse($recentTrades as $trade)
                            <tr class="hover:bg-gray-800/30 transition-colors">
                                <td class="px-4 py-3 text-sm">
                                    {{ $trade->created_at->diffForHumans() }}
                                </td>
                                <td class="px-4 py-3 text-sm">
                                    {{ $trade->pair }}
                                </td>
                                <td class="px-4 py-3">
                                    <span class="px-2 py-1 rounded text-xs font-medium {{ $trade->type === 'Rise' ? 'bg-green-500/10 text-green-500' : 'bg-red-500/10 text-red-500' }}">
                                        {{ $trade->type }}
                                    </span>
                                </td>
                                <td class="px-4 py-3 text-sm">
                                    ${{ number_format($trade->amount, 2) }}
                                </td>
                                <td class="px-4 py-3 text-sm">
                                    @if($trade->status === 'active')
                                        <div class="flex items-center gap-1.5 text-yellow-500">
                                            <i class="fas fa-circle-notch fa-spin text-xs"></i>
                                            <span>Active</span>
                                        </div>
                                    @elseif($trade->status === 'completed')
                                        <span class="text-{{ $trade->result === 'win' ? 'green' : 'red' }}-500">
                                            {{ ucfirst($trade->result) }}
                                        </span>
                                    @else
                                        <span class="text-gray-400">{{ ucfirst($trade->status) }}</span>
                                    @endif
                                </td>
                                <td class="px-4 py-3 text-sm">
                                    @if($trade->status === 'completed')
                                    <span class="font-medium {{ $trade->profit_amount >= 0 ? 'text-green-500' : 'text-red-500' }}">
                                            {{ $trade->profit_amount >= 0 ? '+' : '' }}${{ number_format($trade->profit_amount, 2) }}
                                        </span>
                                    @else
                                        <span class="text-gray-400">--</span>
                                    @endif
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="6" class="px-4 py-8 text-center text-gray-400">
                                    <div class="flex flex-col items-center">
                                        <div class="w-16 h-16 mb-4 rounded-full bg-gray-800/50 flex items-center justify-center">
                                            <i class="fas fa-chart-line text-2xl"></i>
                                        </div>
                                        <p class="mb-2">No trades yet</p>
                                        <p class="text-sm text-gray-500 mb-4">Start your trading journey today</p>
                                        <a href="{{ route('user.trading.trade') }}" class="text-theme-primary hover:underline">
                                            Place Your First Trade
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Fetch prices from database
    async function updatePricesFromDatabase() {
        try {
            const response = await fetch('{{ route("user.trading.prices") }}');
            if (!response.ok) {
                throw new Error('Failed to fetch prices');
            }
            
            const prices = await response.json();
            
            // Update each pair's price display
            Object.keys(prices).forEach(symbol => {
                const data = prices[symbol];
                const symbolKey = symbol.replace('/', '');
                
                // Update price
                const priceElement = document.querySelector(`.price-${symbolKey}`);
                if (priceElement && data.price) {
                    priceElement.textContent = `$${parseFloat(data.price).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}`;
                    priceElement.setAttribute('data-price', data.price);
                }
                
                // Update 24h change
                const changeElement = document.querySelector(`.change-${symbolKey}`);
                if (changeElement && data.change_24h !== null) {
                    const changeValue = parseFloat(data.change_24h);
                    changeElement.textContent = `${changeValue >= 0 ? '+' : ''}${changeValue.toFixed(2)}%`;
                    changeElement.setAttribute('data-change', changeValue);
                    
                    // Update color based on positive/negative
                    changeElement.classList.remove('text-green-500', 'text-red-500');
                    changeElement.classList.add(changeValue >= 0 ? 'text-green-500' : 'text-red-500');
                }
            });
        } catch (error) {
            console.warn('Error fetching prices from database:', error);
        }
    }
    
    // Update prices immediately on page load
    updatePricesFromDatabase();
    
    // Poll for price updates every 30 seconds (matches sync frequency)
    setInterval(updatePricesFromDatabase, 30000);
});

document.addEventListener('DOMContentLoaded', function() {
    const wrapper = document.querySelector('.trading-pairs-wrapper');
    const slides = document.querySelectorAll('.trading-pair-slide');
    const dots = document.querySelectorAll('.trading-pair-dot');
    let currentIndex = 0;
    
    // Touch/Swipe variables
    let startX = 0;
    let endX = 0;
    let isDragging = false;

    function updateCarousel() {
        const slideWidth = slides[0].offsetWidth;
        wrapper.style.transform = `translateX(-${currentIndex * slideWidth}px)`;
        
        // Update dot active state
        dots.forEach((dot, index) => {
            dot.classList.toggle('bg-theme-primary', index === currentIndex);
            dot.classList.toggle('bg-gray-600', index !== currentIndex);
        });
    }

    function handleSwipe() {
        const swipeThreshold = window.innerWidth * 0.2; // 20% of screen width
        const swipeDistance = endX - startX;

        if (Math.abs(swipeDistance) > swipeThreshold) {
            // Swipe right to left
            if (swipeDistance < 0) {
                scrollTradingPairs(1);
            } 
            // Swipe left to right
            else {
                scrollTradingPairs(-1);
            }
        }
    }

    window.scrollTradingPairs = function(direction) {
        currentIndex += direction;
        
        // Wrap around
        if (currentIndex < 0) currentIndex = slides.length - 1;
        if (currentIndex >= slides.length) currentIndex = 0;
        
        updateCarousel();
    }

    window.goToPair = function(index) {
        currentIndex = index;
        updateCarousel();
    }

    // Touch event listeners
    wrapper.addEventListener('touchstart', function(e) {
        startX = e.touches[0].clientX;
        isDragging = true;
    });

    wrapper.addEventListener('touchmove', function(e) {
        if (!isDragging) return;
        endX = e.touches[0].clientX;
    });

    wrapper.addEventListener('touchend', function() {
        if (!isDragging) return;
        handleSwipe();
        isDragging = false;
    });

    // Mouse events for desktop (optional, but improves desktop experience)
    wrapper.addEventListener('mousedown', function(e) {
        startX = e.clientX;
        isDragging = true;
    });

    document.addEventListener('mousemove', function(e) {
        if (!isDragging) return;
        endX = e.clientX;
    });

    document.addEventListener('mouseup', function() {
        if (!isDragging) return;
        handleSwipe();
        isDragging = false;
    });

    // Prevent default drag behavior
    wrapper.addEventListener('dragstart', (e) => e.preventDefault());

    // Auto-scroll every 5 seconds
    setInterval(() => {
        scrollTradingPairs(1);
    }, 5000);

    // Responsive adjustment
    window.addEventListener('resize', updateCarousel);
});
</script>
@endsection



