@extends('layouts.millage')

@section('title', 'Trade History')

@section('content')
    <div class="space-y-6 mt-4 md:mt-1 md:px-1 px-4 md:mb-1 mb-[40px]">
        {{-- Display Errors --}}
        @if ($errors->any())
            <div class="bg-red-500 text-white p-4 rounded mb-4">
                <ul class="list-disc list-inside">
                    @foreach ($errors->all() as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
            </div>
        @endif

        {{-- Header with Summary Stats --}}
        <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {{-- Total Trades --}}
            <div class="bg-theme-secondary rounded-xl p-4">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm sm:text-base text-gray-400">Total Trades</p>
                        <p class="text-lg sm:text-xl font-medium mt-1">{{ number_format($summary['total_trades']) }}</p>
                    </div>
                    <div class="w-10 h-10 flex items-center justify-center rounded-xl bg-theme-primary/10">
                        <i class="fas fa-chart-bar text-theme-primary text-lg sm:text-base"></i>
                    </div>
                </div>
            </div>

            {{-- Total Profit/Loss --}}
            <div class="bg-theme-secondary rounded-xl p-4">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm sm:text-base text-gray-400">Total P/L</p>
                        <p class="text-lg sm:text-xl font-medium mt-1 {{ $summary['total_profit'] >= 0 ? 'text-green-500' : 'text-red-500' }}">
                            {{ $summary['total_profit'] >= 0 ? '+' : '' }}${{ number_format($summary['total_profit'], 2) }}
                        </p>
                    </div>
                    <div @class([
                        'w-10 h-10 flex items-center justify-center rounded-xl',
                        'bg-green-500/10' => $summary['total_profit'] >= 0,
                        'bg-red-500/10' => $summary['total_profit'] < 0,
                    ])>
                        <i @class([
                            'fas fa-dollar-sign text-lg sm:text-base',
                            'text-green-500' => $summary['total_profit'] >= 0,
                            'text-red-500' => $summary['total_profit'] < 0,
                        ])></i>
                    </div>
                </div>
            </div>

            {{-- Wins --}}
            <div class="bg-theme-secondary rounded-xl p-4">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm sm:text-base text-gray-400">Wins</p>
                        <p class="text-lg sm:text-xl font-medium mt-1 text-green-500">{{ number_format($summary['win_count']) }}</p>
                    </div>
                    <div class="w-10 h-10 flex items-center justify-center rounded-xl bg-green-500/10">
                        <i class="fas fa-check text-green-500 text-lg sm:text-base"></i>
                    </div>
                </div>
            </div>

            {{-- Losses --}}
            <div class="bg-theme-secondary rounded-xl p-4">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm sm:text-base text-gray-400">Losses</p>
                        <p class="text-lg sm:text-xl font-medium mt-1 text-red-500">{{ number_format($summary['loss_count']) }}</p>
                    </div>
                    <div class="w-10 h-10 flex items-center justify-center rounded-xl bg-red-500/10">
                        <i class="fas fa-times text-red-500 text-lg sm:text-base"></i>
                    </div>
                </div>
            </div>
        </div>

        {{-- Filters and Table --}}
        <div class="bg-theme-secondary rounded-xl overflow-hidden">
            {{-- Filters --}}
            <div class="p-4 border-b border-gray-800/50">
                <form method="GET" action="{{ route('user.trading.history') }}" class="flex flex-col sm:flex-row sm:flex-wrap items-start sm:items-center gap-4">
                    {{-- Status Filter --}}
                    <div class="w-full sm:w-auto">
                        <label for="status" class="sr-only">Status</label>
                        <select name="status" id="status" class="w-full sm:w-auto bg-theme-tertiary border border-gray-800 rounded-lg px-3 py-2 text-sm text-gray-400 focus:outline-none focus:border-theme-primary">
                            <option value="">All Status</option>
                            <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>Active</option>
                            <option value="completed" {{ request('status') === 'completed' ? 'selected' : '' }}>Completed</option>
                            <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                        </select>
                    </div>

                    {{-- Type Filter --}}
                    <div class="w-full sm:w-auto">
                        <label for="type" class="sr-only">Type</label>
                        <select name="type" id="type" class="w-full sm:w-auto bg-theme-tertiary border border-gray-800 rounded-lg px-3 py-2 text-sm text-gray-400 focus:outline-none focus:border-theme-primary">
                            <option value="">All Types</option>
                            <option value="Rise" {{ request('type') === 'Rise' ? 'selected' : '' }}>Rise</option>
                            <option value="Fall" {{ request('type') === 'Fall' ? 'selected' : '' }}>Fall</option>
                        </select>
                    </div>

                    {{-- Date Range Filter --}}
                    <div class="w-full sm:w-auto">
                        <label for="dateRange" class="sr-only">Date Range</label>
                        <input 
                            type="text" 
                            name="dateRange" 
                            id="dateRange" 
                            value="{{ request('dateRange') }}" 
                            class="w-full sm:w-auto bg-theme-tertiary border border-gray-800 rounded-lg px-3 py-2 text-sm text-gray-400 focus:outline-none focus:border-theme-primary"
                            placeholder="Date Range"
                            autocomplete="off"
                        >
                    </div>

                    {{-- Submit and Reset Filters --}}
                    <div class="w-full sm:w-auto flex flex-col sm:flex-row sm:items-center gap-2">
                        <button 
                            type="submit"
                            class="w-full sm:w-auto px-3 py-2 bg-theme-primary rounded-lg text-sm text-black hover:bg-[#1ab8c1] transition-colors flex items-center justify-center gap-1"
                        >
                            <i class="fas fa-filter"></i>
                            <span>Apply</span>
                        </button>
                        @if(request('status') || request('type') || request('dateRange'))
                            <a 
                                href="{{ route('user.trading.history') }}"
                                class="w-full sm:w-auto px-3 py-2 bg-gray-800/50 rounded-lg text-sm text-gray-400 hover:text-white transition-colors flex items-center justify-center gap-1"
                                aria-label="Reset Filters"
                            >
                                <i class="fas fa-sync-alt"></i>
                                <span>Reset</span>
                            </a>
                        @endif
                    </div>
                </form>
            </div>

            {{-- Table and Cards --}}
            <div class="overflow-x-auto">
                {{-- Table Layout for Small and Above Screens --}}
                <table class="min-w-full divide-y divide-gray-800/50 hidden sm:table">
                    <thead class="bg-theme-tertiary">
                        <tr>
                            <th>
                                <a href="{{ route('user.trading.history', array_merge(request()->all(), ['sort_field' => 'created_at', 'sort_direction' => request('sort_field') === 'created_at' && request('sort_direction') === 'asc' ? 'desc' : 'asc'])) }}" class="px-6 py-3 text-left text-sm font-medium text-gray-400 cursor-pointer flex items-center gap-1">
                                    Time
                                    @if(request('sort_field') === 'created_at')
                                        <i class="fas fa-sort-{{ request('sort_direction') === 'asc' ? 'up' : 'down' }} w-4 h-4"></i>
                                    @endif
                                </a>
                            </th>
                            <th>
                                <a href="{{ route('user.trading.history', array_merge(request()->all(), ['sort_field' => 'type', 'sort_direction' => request('sort_field') === 'type' && request('sort_direction') === 'asc' ? 'desc' : 'asc'])) }}" class="px-6 py-3 text-left text-sm font-medium text-gray-400 cursor-pointer flex items-center gap-1">
                                    Type
                                    @if(request('sort_field') === 'type')
                                        <i class="fas fa-sort-{{ request('sort_direction') === 'asc' ? 'up' : 'down' }} w-4 h-4"></i>
                                    @endif
                                </a>
                            </th>
                            <th>
                                <a href="{{ route('user.trading.history', array_merge(request()->all(), ['sort_field' => 'amount', 'sort_direction' => request('sort_field') === 'amount' && request('sort_direction') === 'asc' ? 'desc' : 'asc'])) }}" class="px-6 py-3 text-left text-sm font-medium text-gray-400 cursor-pointer flex items-center gap-1">
                                    Amount
                                    @if(request('sort_field') === 'amount')
                                        <i class="fas fa-sort-{{ request('sort_direction') === 'asc' ? 'up' : 'down' }} w-4 h-4"></i>
                                    @endif
                                </a>
                            </th>
                            <th class="hidden lg:table-cell">
                                <a href="{{ route('user.trading.history', array_merge(request()->all(), ['sort_field' => 'entry_price', 'sort_direction' => request('sort_field') === 'entry_price' && request('sort_direction') === 'asc' ? 'desc' : 'asc'])) }}" class="px-6 py-3 text-left text-sm font-medium text-gray-400 cursor-pointer flex items-center gap-1">
                                    Entry Price
                                    @if(request('sort_field') === 'entry_price')
                                        <i class="fas fa-sort-{{ request('sort_direction') === 'asc' ? 'up' : 'down' }} w-4 h-4"></i>
                                    @endif
                                </a>
                            </th>
                            <th>
                                <a href="{{ route('user.trading.history', array_merge(request()->all(), ['sort_field' => 'status', 'sort_direction' => request('sort_field') === 'status' && request('sort_direction') === 'asc' ? 'desc' : 'asc'])) }}" class="px-6 py-3 text-left text-sm font-medium text-gray-400 cursor-pointer flex items-center gap-1">
                                    Status
                                    @if(request('sort_field') === 'status')
                                        <i class="fas fa-sort-{{ request('sort_direction') === 'asc' ? 'up' : 'down' }} w-4 h-4"></i>
                                    @endif
                                </a>
                            </th>
                            <th class="hidden md:table-cell">
                                <a href="{{ route('user.trading.history', array_merge(request()->all(), ['sort_field' => 'profit_amount', 'sort_direction' => request('sort_field') === 'profit_amount' && request('sort_direction') === 'asc' ? 'desc' : 'asc'])) }}" class="px-6 py-3 text-left text-sm font-medium text-gray-400 cursor-pointer flex items-center gap-1">
                                    Profit/Loss
                                    @if(request('sort_field') === 'profit_amount')
                                        <i class="fas fa-sort-{{ request('sort_direction') === 'asc' ? 'up' : 'down' }} w-4 h-4"></i>
                                    @endif
                                </a>
                            </th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-800/50">
                        @forelse($trades as $trade)
                            <tr class="hover:bg-gray-800/30 transition-colors">
                                <td class="px-6 py-4 text-sm whitespace-nowrap">
                                    {{ $trade->created_at->format('M d, Y H:i:s') }}
                                </td>
                                <td class="px-6 py-4">
                                    <span @class([
                                        'px-2 py-1 rounded text-xs font-medium',
                                        'bg-green-500/10 text-green-500' => $trade->type === 'Rise',
                                        'bg-red-500/10 text-red-500' => $trade->type === 'Fall',
                                    ])>
                                        {{ $trade->type }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 text-sm hidden md:table-cell">
                                    ${{ number_format($trade->amount, 2) }}
                                </td>
                                <td class="px-6 py-4 text-sm font-mono hidden lg:table-cell">
                                    ${{ number_format($trade->entry_price, 2) }}
                                </td>
                                <td class="px-6 py-4 text-sm">
                                    @if($trade->status === 'active')
                                        <div class="flex items-center gap-1.5 text-yellow-500">
                                            <i class="fas fa-circle-notch fa-spin text-xs"></i>
                                            <span>Active</span>
                                        </div>
                                    @elseif($trade->status === 'completed')
                                        <span @class([
                                            'text-xs font-semibold',
                                            'text-green-500' => $trade->result === 'win',
                                            'text-red-500' => $trade->result === 'loss',
                                        ])>
                                            {{ ucfirst($trade->result) }}
                                        </span>
                                    @else
                                        <span class="text-gray-400">{{ ucfirst($trade->status) }}</span>
                                    @endif
                                </td>
                                <td class="px-6 py-4 text-sm hidden md:table-cell">
                                    @if($trade->status === 'completed')
                                        <span @class([
                                            'font-medium',
                                            'text-green-500' => $trade->profit_amount >= 0,
                                            'text-red-500' => $trade->profit_amount < 0,
                                        ])>
                                            {{ $trade->profit_amount >= 0 ? '+' : '' }}${{ number_format($trade->profit_amount, 2) }}
                                        </span>
                                    @else
                                        <span class="text-gray-400">--</span>
                                    @endif
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="6" class="px-6 py-8 text-center text-gray-400">
                                    <div class="flex flex-col items-center">
                                        <div class="w-16 h-16 mb-4 rounded-full bg-gray-800/50 flex items-center justify-center">
                                            <i class="fas fa-chart-line text-2xl"></i>
                                        </div>
                                        <p class="mb-2">No trades found</p>
                                        <p class="text-sm text-gray-500 mb-4">Try adjusting your filters or start trading</p>
                                        <div class="flex gap-4">
                                            @if(request('status') || request('type') || request('dateRange'))
                                                <a 
                                                    href="{{ route('user.trading.history') }}"
                                                    class="text-theme-primary hover:underline text-sm flex items-center gap-1"
                                                    aria-label="Clear Filters"
                                                >
                                                    <i class="fas fa-sync-alt"></i>
                                                    Clear Filters
                                                </a>
                                            @endif
                                            <a 
                                                href="{{ route('user.trading.trade') }}"
                                                class="text-theme-primary hover:underline text-sm flex items-center gap-1"
                                                aria-label="Start Trading"
                                            >
                                                <i class="fas fa-play-circle"></i>
                                                Start Trading
                                            </a>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>

                {{-- Card Layout for Mobile --}}
                <div class="sm:hidden">
                    @forelse($trades as $trade)
                        <div class="bg-theme-tertiary rounded-lg p-4 mb-4">
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm sm:text-base text-gray-400">Time:</span>
                                <span class="text-sm sm:text-base">{{ $trade->created_at->format('M d, Y H:i:s') }}</span>
                            </div>
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm sm:text-base text-gray-400">Type:</span>
                                <span @class([
                                    'px-2 py-1 rounded text-xs font-medium',
                                    'bg-green-500/10 text-green-500' => $trade->type === 'Rise',
                                    'bg-red-500/10 text-red-500' => $trade->type === 'Fall',
                                ])>
                                    {{ $trade->type }}
                                </span>
                            </div>
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm sm:text-base text-gray-400">Amount:</span>
                                <span class="text-sm sm:text-base">${{ number_format($trade->amount, 2) }}</span>
                            </div>
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm sm:text-base text-gray-400">Entry Price:</span>
                                <span class="text-sm sm:text-base font-mono">${{ number_format($trade->entry_price, 2) }}</span>
                            </div>
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm sm:text-base text-gray-400">Status:</span>
                                @if($trade->status === 'active')
                                    <div class="flex items-center gap-1.5 text-yellow-500">
                                        <i class="fas fa-circle-notch fa-spin text-xs"></i>
                                        <span>Active</span>
                                    </div>
                                @elseif($trade->status === 'completed')
                                    <span @class([
                                        'text-xs font-semibold',
                                        'text-green-500' => $trade->result === 'win',
                                        'text-red-500' => $trade->result === 'loss',
                                    ])>
                                        {{ ucfirst($trade->result) }}
                                    </span>
                                @else
                                    <span class="text-gray-400">{{ ucfirst($trade->status) }}</span>
                                @endif
                            </div>
                            <div class="flex justify-between items-center">
                                <span class="text-sm sm:text-base text-gray-400">Profit/Loss:</span>
                                @if($trade->status === 'completed')
                                    <span @class([
                                        'font-medium',
                                        'text-green-500' => $trade->profit_amount >= 0,
                                        'text-red-500' => $trade->profit_amount < 0,
                                    ])>
                                        {{ $trade->profit_amount >= 0 ? '+' : '' }}${{ number_format($trade->profit_amount, 2) }}
                                    </span>
                                @else
                                    <span class="text-gray-400">--</span>
                                @endif
                            </div>
                        </div>
                    @empty
                        <div class="text-center text-gray-400 p-4">
                            <p>No trades found</p>
                            <a href="{{ route('user.trading.trade') }}" class="text-theme-primary hover:underline">Start Trading</a>
                        </div>
                    @endforelse
                </div>
            </div>

            {{-- Pagination --}}
            @if($trades->hasPages())
                <div class="px-6 py-3 border-t border-gray-800/50">
                    {{ $trades->links() }}
                </div>
            @endif
        </div>
    </div>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize date range picker
            flatpickr("#dateRange", {
                mode: "range",
                dateFormat: "Y-m-d",
                theme: "dark",
                onChange: function(selectedDates, dateStr) {
                    // The input value is automatically updated by Flatpickr
                }
            });
        });
    </script>
@endsection



