<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CryptoAsset extends Model
{
    use HasFactory;

    protected $fillable = [
        'symbol',
        'name',
        'coingecko_id',
        'logo_url',
        'current_price',
        'price_change_24h',
        'price_change_percentage_24h',
        'market_cap',
        'total_volume',
        'high_24h',
        'low_24h',
        'market_cap_rank',
        'is_active',
        'last_updated',
    ];

    protected $casts = [
        'current_price' => 'decimal:8',
        'price_change_24h' => 'decimal:8',
        'price_change_percentage_24h' => 'decimal:4',
        'market_cap' => 'decimal:2',
        'total_volume' => 'decimal:2',
        'high_24h' => 'decimal:8',
        'low_24h' => 'decimal:8',
        'is_active' => 'boolean',
        'last_updated' => 'datetime',
    ];

    /**
     * Get trading pair symbol (e.g., BTC/USDT)
     */
    public function getTradingPairAttribute(): string
    {
        return $this->symbol . '/USDT';
    }

    /**
     * Get Huobi symbol format (e.g., btcusdt)
     */
    public function getHuobiSymbolAttribute(): string
    {
        return strtolower($this->symbol) . 'usdt';
    }

    /**
     * Scope to get only active assets
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to order by market cap rank
     */
    public function scopeByMarketCap($query)
    {
        return $query->orderBy('market_cap_rank', 'asc');
    }
}