<?php

namespace App\Livewire\User\Withdrawal;

use App\Mail\WithdrawalOtpMail;
use App\Mail\WithdrawalRequestMail;
use App\Models\CoinPayment as ModelsCoinPayment;
use App\Models\PaymentMethod;
use App\Models\Settings;
use App\Models\User;
use App\Models\Withdrawal;
use App\Notifications\User\WithdrawalProcessedNotification;
use App\Notifications\User\WithdrawalSuccessNotification;
use App\Traits\Coinpayment;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Mail;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class CompleteWithdrawal extends Component
{
    use LivewireAlert;
    use Coinpayment;

    public PaymentMethod $method;
    public $amount;
    public $otp;
    public $wallet_address;
    public $useotp = false;
    public $details;

    public function mount()
    {
        $this->useotp = Settings::select('id', 'enable_withdrawal_otp')->find(1)->enable_withdrawal_otp;
        $this->wallet_address = match ($this->method->name) {
            'Bitcoin' => auth()->user()->btc_address,
            'Ethereum' => auth()->user()->eth_address,
            'Litecoin' => auth()->user()->ltc_address,
            'USDT' => auth()->user()->usdt_address,
            default => null
        };
    }

    public function render()
{
    $template = Settings::select('theme')->find(1)->theme;
    $settings = Settings::find(1); // Pass settings explicitly

    return view("{$template}.withdrawal.complete-withdrawal", [
        'settings' => $settings,
    ]);
}

    protected function rules()
    {
    $min = $this->method->minimum;
    $max = $this->method->maximum;

    return [
        'amount' => "required|numeric|min:{$min}|max:{$max}",
        ];
    }
    
    protected $messages = [
    'amount.required' => 'Amount is required.',
    'amount.numeric' => 'Amount must be a number.',
    'amount.min' => 'The minimum withdrawal amount is :min.',
    'amount.max' => 'The maximum withdrawal amount is :max.',
    ];
    
    public function updatedAmount()
    {
    $this->validateOnly('amount');
    }


     public $statusMessage = '';

    public function requestOtp(): void
    {
    $code = strtoupper($this->RandomStringGenerator(5));
    $user = User::find(auth()->user()->id);
    $user->withdrawal_otp = $code;
    $user->withdrawal_otp_expired_at = now()->addMinutes(5);
    $user->save();

    Mail::to($user->email)->send(new WithdrawalOtpMail($user->name, $code));

    $this->statusMessage = __('t.otp_sent_success');
    }

    public function save()
    {
        
    $this->validate();
    
    $settings = Settings::where('id', '1')->first();
    $user = User::find(auth()->user()->id);

    if ($this->useotp && ($this->otp !== $user->withdrawal_otp || $user->withdrawal_otp_expired_at < now())) {
    return redirect()->back()->with('message', __('t.otp_invalid_or_expired'));
}

if ($settings->enable_kyc && $user->account_verify !== 'verified') {
    return redirect()->back()->with('message', __('t.kyc_required_for_withdrawal'));
}

$charges = $this->method->charges_type === 'percentage' 
    ? floatval($this->amount) * $this->method->charges_amount / 100 
    : $this->method->charges_amount;

$to_withdraw = floatval($this->amount) + $charges;

if ($user->account_bal < $to_withdraw) {
    return redirect()->back()->with('message', __('t.insufficient_balance_withdrawal'));
}

if ($this->method->name === 'Bitcoin' && empty($user->btc_address)) {
    return redirect()->back()->with('message', __('t.setup_bitcoin_wallet'));
}
if ($this->method->name === 'Ethereum' && empty($user->eth_address)) {
    return redirect()->back()->with('message', __('t.setup_ethereum_wallet'));
}
if ($this->method->name === 'Litecoin' && empty($user->ltc_address)) {
    return redirect()->back()->with('message', __('t.setup_litecoin_wallet'));
}
if ($this->method->name === 'USDT' && empty($user->usdt_address)) {
    return redirect()->back()->with('message', __('t.setup_usdt_wallet'));
}


    // Auto Withdrawal (for crypto)
    if ($settings->withdrawal_option === 'auto' && $this->method->methodtype === 'crypto') {
        try {
            $response = $this->cpwithdraw(floatval($this->amount), $this->method->coin, $this->wallet_address);

            if ($response['status'] === 'error') {
                return redirect()->back()->with('message', $response['message']);
            }

            // Save withdrawal info
            $dp = Withdrawal::create([
                'user_id' => $user->id,
                'txn_id' => $response['txn_id'],
                'amount' => $this->amount,
                'to_deduct' => $to_withdraw,
                'payment_mode' => "{$this->method->name}(CoinPayments)",
                'status' => 'Processed',
            ]);

            // Debit user account balance
            $user->account_bal -= $to_withdraw;
            $user->withdrawal_otp = null;
            $user->withdrawal_otp_expired_at = null;
            $user->save();

            // Notify user
            session()->flash('success', __('t.withdrawal_processed'));

            // Email notifications
            if ($settings->receive_withdrawal_email) {
                dispatch(function () use ($settings, $dp) {
                    Mail::to($settings->notifiable_email)->send(new WithdrawalRequestMail($dp));
                })->afterResponse();
            }
            return redirect()->route('user.withdrawals.history');

        } catch (\Throwable $th) {
            return redirect()->back()->with('message', $th->getMessage());
        }
    }

    // Save withdrawal info (manual processing)
    $dp = Withdrawal::create([
        'user_id' => $user->id,
        'amount' => $this->amount,
        'to_deduct' => $to_withdraw,
        'payment_mode' => $this->method->name,
        'status' => 'pending',
        'paydetails' => $this->wallet_address . $this->details,
    ]);

    // Send mail to admin
    if ($settings->receive_withdrawal_email) {
        dispatch(function () use ($settings, $dp) {
            Mail::to($settings->notifiable_email)->send(new WithdrawalRequestMail($dp));
        })->afterResponse();
    }

    try {
    if ($settings->send_withdrawal_email) {
        $message = __('t.withdrawal_request_submitted', [
            'currency' => $settings->currency,
            'amount'   => $this->amount,
            'method'   => $this->method->name
        ]);

        dispatch(function () use ($dp, $user, $message) {
            $user->notify(new WithdrawalSuccessNotification($dp, $message));
        })->afterResponse();
    }
} catch (\Throwable $th) {
    return redirect()->back()->with('message', __('t.error_try_again'));
}


    // Reset fields
    $this->reset([
        'amount',
        'wallet_address',
        'details',
        'otp',
        'useotp',
    ]);

    Cache::forget('chart_pendwithdraw');
    Cache::forget('total_withdrawn');

    return redirect()->route('user.transactions.withdrawal')->with('success', __('t.action_successful_processing'));
}


    private function RandomStringGenerator($n)
    {
        $generated_string = '';
        $domain = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890';
        $len = strlen($domain);
        for ($i = 0; $i < $n; $i++) {
            $index = rand(0, $len - 1);
            $generated_string .= $domain[$index];
        }
        // Return the random generated string
        return $generated_string;
    }
}
