<?php

namespace App\Livewire\Admin\Withdrawal;

use App\Models\PaymentMethod;
use App\Models\Settings;
use App\Models\Withdrawal;
use App\Models\Transaction;
use App\Notifications\User\WithdrawalFailedNotification;
use App\Notifications\User\WithdrawalProcessedNotification;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Gate;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('layouts.admin')]
class ProcessWithdrawal extends Component
{
    use LivewireAlert;
    
    public Withdrawal $withdrawal;
    public PaymentMethod $method;
    public $action = 'Paid';
    
    public function mount($id)
    {
        $this->withdrawal = Withdrawal::with('user')->find($id);
        $this->method = PaymentMethod::where('name', $this->withdrawal->payment_mode)->first();
    }
    
    public function render()
    {
        return view('livewire.admin.withdrawal.process-withdrawal', [
            'settings' => Cache::remember('site_settings', now()->addHour(), function () {
                return Settings::first();
            }),
        ]);
    }
    
    // Note: Method name has typo but must match the blade template's wire:submit and wire:target
    public function processWithrdawal()
    {
        Gate::authorize('process withdrawals');
        
        try {
            $settings = Cache::get('site_settings');
            
            if ($this->action === 'Paid') {
                // Approve and process the withdrawal
                if ($settings->deduction_option === 'AdminApprove') {
                    // Update user's account balance
                    $this->withdrawal->user->account_bal -= $this->withdrawal->to_deduct;
                    $this->withdrawal->user->save();
                }
                
                // Update withdrawal status
                $this->withdrawal->status = 'processed';
                $this->withdrawal->save();
                
                // Record the transaction
                Transaction::create([
                    'user_id' => $this->withdrawal->user_id,
                    'narration' => "Withdrawal of {$settings->currency} {$this->withdrawal->amount} processed",
                    'amount' => $this->withdrawal->amount,
                    'type' => 'Debit',
                ]);
                
                // Send notification to user if enabled
                if ($settings->send_withdrawal_email) {
                    $message = "Your withdrawal request of {$settings->currency}{$this->withdrawal->amount} was approved and funds have been sent to your selected account";
                    try {
                        $this->withdrawal->user->notify(new WithdrawalProcessedNotification($this->withdrawal, $message));
                    } catch (\Throwable $e) {
                        \Log::error('Withdrawal processed notify failed', [
                            'withdrawal_id' => $this->withdrawal->id,
                            'user_id' => $this->withdrawal->user_id,
                            'error' => $e->getMessage(),
                        ]);
                    }
                }
            } else {
                // Cancel the withdrawal
                if ($settings->deduction_option === 'userRequest') {
                    // Restore user's account balance if it was deducted on request
                    $this->withdrawal->user->account_bal += $this->withdrawal->to_deduct;
                    $this->withdrawal->user->save();
                }
                
                // Update withdrawal status
                $this->withdrawal->status = 'cancelled';
                $this->withdrawal->save();
                
                // Send notification to user if enabled
                if ($settings->send_withdrawal_email) {
                    $message = "Your withdrawal request of {$settings->currency}{$this->withdrawal->amount} has been rejected";
                    try {
                        $this->withdrawal->user->notify(new WithdrawalFailedNotification($this->withdrawal, $message));
                    } catch (\Throwable $e) {
                        \Log::error('Withdrawal rejected notify failed', [
                            'withdrawal_id' => $this->withdrawal->id,
                            'user_id' => $this->withdrawal->user_id,
                            'error' => $e->getMessage(),
                        ]);
                    }
                }
            }
            
            // Clear relevant caches
            Cache::forget('total_withdrawn');
            Cache::forget('chart_pendwithdraw');
            
            $this->flash(message: 'Action Successful!', redirect: route('admin.manageWithdrawal'));
        } catch (\Throwable $th) {
            $this->alert(type: 'error', message: $th->getMessage());
        }
    }
}