<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Trade;
use App\Models\User;
use App\Models\Settings;
use App\Models\CryptoAsset;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Validation\ValidationException;
use App\Jobs\ProcessTrade;
use Carbon\Carbon;

class TradingController extends Controller
{

    private $pairToCoinGeckoId = [];
    private $supportedQuoteCurrencies = ['USDT'];
    private $excludedCoins = ['wrapped-steth', 'weth', 'steth'];

    private function getHuobiPairs()
    {
        try {
            $response = Http::get('https://api.huobi.pro/v1/common/symbols');
            if (!$response->successful()) {
                throw new \Exception('Failed to fetch Huobi symbols');
            }

            return collect($response->json()['data'])
                ->where('state', 'online')
                ->where('quote-currency', strtolower($this->supportedQuoteCurrencies[0]))
                ->pluck('base-currency')
                ->map(fn($base) => strtoupper($base) . '/' . $this->supportedQuoteCurrencies[0])
                ->toArray();
        } catch (\Exception $e) {
            Log::error('Huobi pairs fetch failed: ' . $e->getMessage());
            return [];
        }
    }

    private function getCoinGeckoTopCoins()
    {
        try {
            $response = Http::get('https://api.coingecko.com/api/v3/coins/markets', [
                'vs_currency' => 'usd',
                'order' => 'market_cap_desc',
                'per_page' => 250,
                'page' => 1,
                'sparkline' => false
            ]);

            if (!$response->successful()) {
                throw new \Exception('Failed to fetch CoinGecko market data');
            }

            return collect($response->json())
                ->reject(function ($coin) {
                    return in_array($coin['id'], $this->excludedCoins) ||
                        str_starts_with(strtolower($coin['id']), 'wrapped-') ||
                        str_starts_with(strtolower($coin['symbol']), 'w') ||
                        strtoupper($coin['symbol']) === 'USDT';
                })
                ->mapWithKeys(function ($coin) {
                    return [
                        strtoupper($coin['symbol']) => [
                            'id' => $coin['id'],
                            'name' => $coin['name'],
                            'image' => $coin['image'],
                            'market_cap_rank' => $coin['market_cap_rank']
                        ]
                    ];
                })
                ->toArray();
        } catch (\Exception $e) {
            Log::error('CoinGecko data fetch failed: ' . $e->getMessage());
            return [];
        }
    }

    private function initializeTradingPairs()
    {
        $cacheKey = 'trading_pairs_mapping';
        if (Cache::has($cacheKey)) {
            $this->pairToCoinGeckoId = Cache::get($cacheKey);
            return;
        }

        try {
            $huobiPairs = $this->getHuobiPairs();
            $coinGeckoData = $this->getCoinGeckoTopCoins();

            // Match Huobi pairs with CoinGecko data
            $this->pairToCoinGeckoId = collect($huobiPairs)
                ->filter(function ($pair) use ($coinGeckoData) {
                    $baseSymbol = explode('/', $pair)[0];
                    return isset($coinGeckoData[$baseSymbol]);
                })
                ->mapWithKeys(function ($pair) use ($coinGeckoData) {
                    $baseSymbol = explode('/', $pair)[0];
                    return [$pair => $coinGeckoData[$baseSymbol]['id']];
                })
                ->toArray();

            Cache::put($cacheKey, $this->pairToCoinGeckoId, now()->addHours(24));
        } catch (\Exception $e) {
            Log::error('Trading pairs initialization failed: ' . $e->getMessage());
            $this->pairToCoinGeckoId = [
                'BTC/USDT' => 'bitcoin',
                'ETH/USDT' => 'ethereum'
            ];
        }
    }

    private function fetchPairImages()
    {
        try {
            $colors = [
                'text-blue-500',
                'text-green-500',
                'text-yellow-500',
                'text-red-500',
                'text-purple-500',
                'text-indigo-500',
                'text-pink-500',
                'text-teal-500',
                'text-orange-500',
                'text-cyan-500'
            ];

            // Fetch active crypto assets from database
            $assets = CryptoAsset::active()
                ->byMarketCap()
                ->get();

            $colorIndex = 0;
            $pairData = $assets->map(function ($asset) use ($colors, &$colorIndex) {
                return [
                    'symbol' => $asset->trading_pair,
                    'name' => $asset->name,
                    'icon' => $asset->logo_url ?? asset('images/default.png'),
                    'color' => $colors[$colorIndex++ % count($colors)],
                    'market_cap_rank' => $asset->market_cap_rank,
                    'price' => $asset->current_price,
                    'change_24h' => $asset->price_change_percentage_24h
                ];
            })->toArray();

            return $pairData;
        } catch (\Exception $e) {
            Log::error('Pair images fetch failed: ' . $e->getMessage());
            return [];
        }
    }

    public function index()
    {
        $user = auth()->user();
        $settings = Settings::first();

        // Get user's trading statistics
        $stats = [
            'total_trades' => Trade::where('user_id', $user->id)->count(),
            'win_rate' => Trade::where('user_id', $user->id)
                ->whereNotNull('result')
                ->count() > 0
                ? (Trade::where('user_id', $user->id)
                    ->where('result', 'win')
                    ->count() / Trade::where('user_id', $user->id)
                    ->whereNotNull('result')
                    ->count()) * 100
                : 0,
            'total_profit' => Trade::where('user_id', $user->id)
                ->where('status', 'completed')
                ->sum('profit_amount'),
            'best_trade' => Trade::where('user_id', $user->id)
                ->where('status', 'completed')
                ->where('result', 'win')
                ->orderBy('profit_amount', 'desc')
                ->first()
        ];

        // Fetch pairs with images and data
        $pairs = array_values($this->fetchPairImages());

        // Get recent trades
        $recentTrades = Trade::where('user_id', $user->id)
            ->latest()
            ->limit(5)
            ->get();

        return view("{$settings->theme}.trading-page.index", compact(
            'stats',
            'pairs',
            'recentTrades',
            'user'
        ));
    }


    public function trade(Request $request, $pair = null)
    {
        // Set default pair if none provided
        $pair = $pair ?? 'BTCUSDT';

        // Extract symbol from pair (remove USDT)
        $symbol = strtoupper(str_replace('USDT', '', $pair));
        
        // Format pair for display (add slash between currency pairs)
        $displayPair = $symbol . '/USDT';

        $user = auth()->user();
        $settings = Settings::first();

        // Fetch pairs with images and data
        $pairData = $this->fetchPairImages();

        // Convert pairData to the format needed for the view
        $pairs = array_values($pairData);

        // Get asset data from database
        $asset = CryptoAsset::where('symbol', $symbol)->first();

        // Initialize market stats from database
        if ($asset) {
            $stats = [
                'volume' => $asset->total_volume,
                'high' => $asset->high_24h,
                'low' => $asset->low_24h,
                'close' => $asset->current_price,
                'change' => $asset->price_change_24h,
                'changePercent' => $asset->price_change_percentage_24h
            ];
        } else {
            // Fallback if asset not found
            $stats = [
                'volume' => 0,
                'high' => 0,
                'low' => 0,
                'close' => 0,
                'change' => 0,
                'changePercent' => 0
            ];
            Log::warning("Asset not found in database: {$symbol}");
        }

        // Fetch recent trades for the user and specific pair
        $recentTrades = Trade::where('user_id', $user->id)
            ->where('pair', $pair)
            ->latest()
            ->limit(5)
            ->get();

        // Calculate total profit from completed trades
        $totalProfit = Trade::where('user_id', $user->id)
            ->where('status', 'completed')
            ->sum('profit_amount');

        // Prepare data to pass to the view
        $data = [
            'user' => $user,
            'pair' => $displayPair,
            'symbol' => $pair,
            'pairs' => $pairs,
            'wallet' => $user->account_bal,
            'profit_percentage' => $settings->profit_percent,
            'currentPrice' => $stats['close'],
            'stats' => $stats,
            'timeframes' => [
                '1' => '1 Minute',
                '5' => '5 Minutes',
                '15' => '15 Minutes',
                '30' => '30 Minutes',
                '60' => '1 Hour',
                '240' => '4 Hours',
                '1D' => '1 Day'
            ],
            'durations' => ['1m', '5m', '15m', '30m', '1h', '4h', '1d'],
            'recentTrades' => $recentTrades,
            'totalProfit' => $totalProfit
        ];

        return view("{$settings->theme}.trading-page.trade", $data);
    }

    public function getKlineData(Request $request)
    {
        $request->validate([
            'symbol' => 'required|string',
            'period' => 'required|string|in:1min,5min,15min,30min,60min,4hour,1day',
            'size' => 'integer|max:2000'
        ]);

        try {
            $response = Http::get('https://api.huobi.pro/market/history/kline', [
                'symbol' => strtolower($request->symbol),
                'period' => $request->period,
                'size' => $request->size ?? 200
            ]);

            if ($response->successful() && $response->json()['status'] === 'ok') {
                return response()->json([
                    'success' => true,
                    'data' => $response->json()['data']
                ]);
            }

            throw new \Exception('Failed to fetch kline data');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch kline data'
            ], 500);
        }
    }

    // Add method to get market depth
    public function getDepth(Request $request)
    {
        $request->validate([
            'symbol' => 'required|string',
            'type' => 'required|string|in:step0,step1,step2,step3,step4,step5'
        ]);

        try {
            $response = Http::get('https://api.huobi.pro/market/depth', [
                'symbol' => strtolower($request->symbol),
                'type' => $request->type
            ]);

            if ($response->successful() && $response->json()['status'] === 'ok') {
                return response()->json([
                    'success' => true,
                    'data' => $response->json()['tick']
                ]);
            }

            throw new \Exception('Failed to fetch depth data');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch depth data'
            ], 500);
        }
    }

    public function history(Request $request)
    {
        // Authenticate the user
        $user = auth()->user();

        // Retrieve settings (ensure it exists)
        $settings = Settings::first();

        if (!$settings) {
            return redirect()->back()->withErrors('Settings not found. Please contact support.');
        }

        // Initialize the trade query for the authenticated user
        $query = Trade::where('user_id', $user->id);

        // Apply Status Filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Apply Type Filter
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Apply Date Range Filter
        if ($request->filled('dateRange')) {
            $dates = explode(' - ', $request->dateRange);
            if (count($dates) === 2) {
                try {
                    $start = Carbon::parse($dates[0])->startOfDay();
                    $end = Carbon::parse($dates[1])->endOfDay();
                    $query->whereBetween('created_at', [$start, $end]);
                } catch (\Exception $e) {
                    // Handle invalid date formats
                    return redirect()->back()->withErrors('Invalid date range format.');
                }
            }
        }

        // Apply Sorting
        $allowedSortFields = ['created_at', 'type', 'amount', 'entry_price', 'status', 'profit_amount'];
        $sortField = in_array($request->sort_field, $allowedSortFields) ? $request->sort_field : 'created_at';
        $sortDirection = $request->sort_direction === 'asc' ? 'asc' : 'desc';

        $trades = $query->orderBy($sortField, $sortDirection)->paginate(10)->appends($request->except('page'));

        // Define summary statistics based on filtered trades
        $summary = [
            'total_trades' => $query->count(),
            'total_profit' => Trade::where('user_id', $user->id)
                ->where('status', 'completed')
                ->sum('profit_amount'),
            'win_count' => Trade::where('user_id', $user->id)
                ->where('status', 'completed')
                ->where('result', 'win')
                ->count(),
            'loss_count' => Trade::where('user_id', $user->id)
                ->where('status', 'completed')
                ->where('result', 'loss')
                ->count(),
        ];

        return view("{$settings->theme}.trading-page.history", compact('summary', 'trades'));
    }


    public function getTrades(Request $request)
    {
        try {
            $settings = Settings::select('theme')->first();
            $query = Trade::where('user_id', auth()->id());

            // Apply filters
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }
            if ($request->has('type')) {
                $query->where('type', $request->type);
            }
            if ($request->has('dateRange')) {
                $dates = explode(' - ', $request->dateRange);
                if (count($dates) == 2) {
                    $start = Carbon::parse($dates[0])->startOfDay();
                    $end = Carbon::parse($dates[1])->endOfDay();
                    $query->whereBetween('created_at', [$start, $end]);
                }
            }

            // Apply sorting
            $sortField = $request->get('sort_field', 'created_at');
            $sortDirection = $request->get('sort_direction', 'desc');
            $query->orderBy($sortField, $sortDirection);

            // Paginate results
            $trades = $query->paginate(15);

            // Always return JSON
            return response()->json([
                'success' => true,
                'html' => view("trading-page.partials.trade-list", compact('trades'))->render(),
                'pagination' => view('components.pagination', ['paginator' => $trades])->render()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve trades: ' . $e->getMessage()
            ], 500);
        }
    }

    public function placeTrade(Request $request)
    {
        try {
            // Validate request
            $validated = $request->validate([
                'amount' => 'required|numeric|min:1',
                'duration' => 'required|in:1m,5m,15m,30m,1h,4h,1d',
                'type' => 'required|in:Rise,Fall',
                'entry_price' => 'required|numeric',
                'pair' => 'required|string',
                'profit_percentage' => 'required|numeric'
            ]);

            $user = auth()->user();

            // Check balance
            if ($request->amount > $user->account_bal) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance'
                ], 422);
            }

            // Get current market price from database
            $symbol = strtoupper(str_replace(['/', 'USDT'], '', $request->pair));
            $asset = CryptoAsset::where('symbol', $symbol)->first();

            if (!$asset) {
                return response()->json([
                    'success' => false,
                    'message' => 'Asset not found'
                ], 422);
            }

            $marketPrice = $asset->current_price;
            
            // Verify price hasn't moved more than 1% from client's entry price
            $priceDifference = abs($marketPrice - $request->entry_price) / $marketPrice * 100;
            
            if ($priceDifference > 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Price has changed significantly. Please refresh and try again.'
                ], 422);
            }

            // Process trade
            DB::transaction(function () use ($request, $user, $marketPrice) {
                // Create trade record
                $trade = Trade::create([
                    'user_id' => $user->id,
                    'pair' => $request->pair,
                    'amount' => $request->amount,
                    'type' => $request->type,
                    'entry_price' => $marketPrice, // Use actual market price
                    'profit_percentage' => $request->profit_percentage,
                    'duration' => $request->duration,
                    'status' => 'active'
                ]);

                // Deduct from user's balance
                $user->decrement('account_bal', $request->amount);

                // Schedule trade completion
                dispatch(new ProcessTrade($trade))->delay(
                    $this->getDurationInSeconds($request->duration)
                );

                // Log trade placement
                Log::info('Trade placed', [
                    'trade_id' => $trade->id,
                    'user_id' => $user->id,
                    'amount' => $request->amount,
                    'type' => $request->type,
                    'entry_price' => $marketPrice
                ]);
            });

            return response()->json([
                'success' => true,
                'message' => 'Trade placed successfully',
                'new_balance' => auth()->user()->fresh()->account_bal
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Trade placement error', [
                'user_id' => auth()->id(),
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to place trade. Please try again.'
            ], 500);
        }
    }

    public function getRecentTrades()
    {
        try {
            $trades = Trade::where('user_id', auth()->id())
                ->latest()
                ->limit(5)
                ->get()
                ->map(function ($trade) {
                    return [
                        'id' => $trade->id,
                        'created_at' => $trade->created_at->diffForHumans(),
                        'type' => $trade->type,
                        'amount' => number_format($trade->amount, 2),
                        'entry_price' => number_format($trade->entry_price, 2),
                        'status' => $trade->status,
                        'result' => $trade->result,
                        'profit_amount' => $trade->profit_amount ? number_format($trade->profit_amount, 2) : null
                    ];
                });

            return response()->json([
                'success' => true,
                'trades' => $trades
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve recent trades',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getMarketData(Request $request)
    {
        try {
            // Extract symbol from request (e.g., btcusdt -> BTC)
            $symbol = strtoupper(str_replace('usdt', '', $request->symbol));
            
            // Get asset from database
            $asset = CryptoAsset::where('symbol', $symbol)->first();

            if (!$asset) {
                throw new \Exception('Asset not found');
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'price' => $asset->current_price,
                    'high' => $asset->high_24h,
                    'low' => $asset->low_24h,
                    'volume' => $asset->total_volume,
                    'change' => $asset->price_change_24h,
                    'changePercent' => $asset->price_change_percentage_24h,
                    'close' => $asset->current_price,
                    'open' => $asset->current_price - $asset->price_change_24h,
                    'vol' => $asset->total_volume
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch market data: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getPrices()
    {
        try {
            // Get all active crypto assets
            $assets = CryptoAsset::active()->get();
            
            $prices = [];
            foreach ($assets as $asset) {
                $symbol = $asset->trading_pair; // Returns "BTC/USDT" format
                $prices[$symbol] = [
                    'price' => $asset->current_price,
                    'change_24h' => $asset->price_change_percentage_24h,
                    'high_24h' => $asset->high_24h,
                    'low_24h' => $asset->low_24h,
                    'volume' => $asset->total_volume,
                    'last_updated' => $asset->last_updated
                ];
            }
            
            return response()->json($prices);
        } catch (\Exception $e) {
            Log::error('Error fetching prices: ' . $e->getMessage());
            return response()->json([
                'error' => 'Unable to fetch prices'
            ], 500);
        }
    }

    private function getDurationInSeconds($duration)
    {
        return match ($duration) {
            '1m' => 60,
            '5m' => 300,
            '15m' => 900,
            '30m' => 1800,
            '1h' => 3600,
            '4h' => 14400,
            '1d' => 86400,
            default => 60,
        };
    }
}
