<?php

namespace App\Console\Commands;

use App\Jobs\ProcessTrade;
use App\Models\Trade;
use Carbon\Carbon;
use Illuminate\Console\Command;

class ProcessStuckTrades extends Command
{
    protected $signature = 'trades:process-stuck';
    protected $description = 'Process stuck active trades that should have completed';

    public function handle()
    {
        $this->info('Checking for stuck trades...');

        // Get all active trades
        $trades = Trade::where('status', 'active')->get();

        if ($trades->isEmpty()) {
            $this->info('No stuck trades found.');
            return 0;
        }

        $processed = 0;
        foreach ($trades as $trade) {
            $durationInSeconds = $this->getDurationInSeconds($trade->duration);
            $shouldCompleteAt = $trade->created_at->addSeconds($durationInSeconds);

            if (now()->greaterThan($shouldCompleteAt)) {
                $this->info("Processing stuck trade #{$trade->id} (Duration: {$trade->duration}, Created: {$trade->created_at})");
                
                try {
                    // Process the trade immediately
                    $job = new ProcessTrade($trade);
                    $job->handle();
                    
                    $this->info("✓ Trade #{$trade->id} processed successfully. Result: {$trade->result}");
                    $processed++;
                } catch (\Exception $e) {
                    $this->error("✗ Failed to process trade #{$trade->id}: {$e->getMessage()}");
                }
            } else {
                $minutesRemaining = now()->diffInMinutes($shouldCompleteAt, false);
                $this->comment("Trade #{$trade->id} will complete in {$minutesRemaining} minute(s)");
            }
        }

        $this->info("\nProcessed {$processed} stuck trade(s).");
        return 0;
    }

    private function getDurationInSeconds($duration)
    {
        return match ($duration) {
            '1m' => 60,
            '5m' => 300,
            '15m' => 900,
            '30m' => 1800,
            '1h' => 3600,
            '4h' => 14400,
            '1d' => 86400,
            default => 60,
        };
    }
}
