<?php

namespace App\Console\Commands;

use App\Models\Settings;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class MigrateAssetsToPublic extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'assets:migrate-to-public';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Migrate logo and favicon from storage to public assets directory';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting migration of assets to public directory...');
        
        // Ensure assets directory exists
        if (!File::exists(public_path('assets/images'))) {
            File::makeDirectory(public_path('assets/images'), 0755, true);
            $this->info('Created assets/images directory');
        }
        
        $settings = Settings::find(1);
        $updated = false;
        
        // Migrate logo
        if ($settings->logo && str_starts_with($settings->logo, 'settings/')) {
            $oldLogoPath = storage_path('app/public/' . $settings->logo);
            
            if (File::exists($oldLogoPath)) {
                $logoExtension = pathinfo($oldLogoPath, PATHINFO_EXTENSION);
                $newLogoName = 'logo_' . time() . '.' . $logoExtension;
                $newLogoPath = public_path('assets/images/' . $newLogoName);
                
                if (File::copy($oldLogoPath, $newLogoPath)) {
                    $settings->logo = 'assets/images/' . $newLogoName;
                    $updated = true;
                    $this->info("✓ Migrated logo: {$settings->logo}");
                } else {
                    $this->error("✗ Failed to copy logo file");
                }
            } else {
                $this->warn("Logo file not found in storage: {$oldLogoPath}");
            }
        } else if ($settings->logo && !str_starts_with($settings->logo, 'assets/')) {
            $this->info("Logo already migrated or not in expected storage format");
        }
        
        // Migrate favicon
        if ($settings->favicon && str_starts_with($settings->favicon, 'settings/')) {
            $oldFaviconPath = storage_path('app/public/' . $settings->favicon);
            
            if (File::exists($oldFaviconPath)) {
                $faviconExtension = pathinfo($oldFaviconPath, PATHINFO_EXTENSION);
                $newFaviconName = 'favicon_' . time() . '.' . $faviconExtension;
                $newFaviconPath = public_path('assets/images/' . $newFaviconName);
                
                if (File::copy($oldFaviconPath, $newFaviconPath)) {
                    $settings->favicon = 'assets/images/' . $newFaviconName;
                    $updated = true;
                    $this->info("✓ Migrated favicon: {$settings->favicon}");
                } else {
                    $this->error("✗ Failed to copy favicon file");
                }
            } else {
                $this->warn("Favicon file not found in storage: {$oldFaviconPath}");
            }
        } else if ($settings->favicon && !str_starts_with($settings->favicon, 'assets/')) {
            $this->info("Favicon already migrated or not in expected storage format");
        }
        
        // Save updated settings
        if ($updated) {
            $settings->save();
            $this->info('✓ Updated database with new asset paths');
            
            // Clear cache
            \Illuminate\Support\Facades\Cache::forget('site_settings');
            $this->info('✓ Cleared settings cache');
        }
        
        $this->info('Asset migration completed!');
        
        if ($updated) {
            $this->warn('NOTE: You can now safely remove old files from storage/app/public/settings/ if migration was successful');
        }
        
        return 0;
    }
}
